package ga.core.individual.population;

import ga.core.GA;
import ga.core.algorithm.util.RandomSingleton;
import ga.core.evaluation.IFitnessEvaluator;
import ga.core.individual.IIndividual;
import ga.core.individual.IIndividualFactory;
import ga.core.individual.IndividualList;
import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Random;

/**
 * Population implementation that uses an {@link ArrayList}.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class ArrayListPopulation<T extends IIndividual<T>> implements
    IPopulation<T> {
  private final IndividualList<T> pop = new IndividualList<T>();
  private final IIndividualFactory<T> factory;
  private IFitnessEvaluator<T> evaluator;

  private boolean allowDuplicates;

  private final Random rnd = RandomSingleton.getRandom();

  private int initIndividualCount = 40;

  /**
   * Creates a new array list population.
   * 
   * @param factory
   *          The individual factory.
   * @param initIndividualCount
   *          Init size of the population.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public ArrayListPopulation(final IIndividualFactory<T> factory,
      final int initIndividualCount) {
    this(factory, initIndividualCount, true);
  }

  /**
   * Creates a new array list population.
   * 
   * @param factory
   *          The individual factory.
   * @param initIndividualCount
   *          Init size of the population.
   * @param allowDuplicates
   *          Allow duplicate individuals if <code>true</code> but prevent them
   *          otherwise.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public ArrayListPopulation(final IIndividualFactory<T> factory,
      final int initIndividualCount, final boolean allowDuplicates) {
    this.factory = factory;
    this.initIndividualCount = initIndividualCount;
    this.allowDuplicates = allowDuplicates;
  }

  @Override
  public void setEvaluator(final IFitnessEvaluator<T> evaluator) {
    this.evaluator = evaluator;
  }

  @Override
  public void initRandomly(final IValidator<T> validator,
      final GAContext context) {

    try {
      @SuppressWarnings("unchecked")
      final T template = (T) context.get(GA.KEY_INIT_INDIVIDUAL);

      if (template != null && validator.isValid(template, context)) {
        final int percentage = context.getInt(
            GA.KEY_INIT_INDIVIDUAL_PERCENTAGE, 1);

        final int count = Math.min(
            Math.max((percentage * initIndividualCount) / 100, 1), 100);

        for (int i = 0; i < count; i++) {
          pop.add(template);
        }
      }
    } catch (final Exception e) {
      // ignore
    }

    while (pop.size() < initIndividualCount) {
      final T ind = factory.newIndividual(context);

      do {
        ind.initRandomly();
      } while (validator != null && !validator.isValid(ind, context));

      pop.add(ind);
    }
  }

  @Override
  public IndividualList<T> getIndividuals() {
    return pop;
  }

  @Override
  public void addIndividuals(final T... individuals) {
    for (final T ind : individuals) {
      if (allowDuplicates || !pop.contains(ind)) {
        pop.add(ind);
      }
    }
  }

  @Override
  public void addIndividual(final T individual) {
    if (allowDuplicates || !pop.contains(individual)) {
      pop.add(individual);
    }
  }

  @Override
  public void addIndividuals(final IndividualList<T> individuals) {
    for (final T ind : individuals) {
      if (allowDuplicates || !pop.contains(ind)) {
        pop.add(ind);
      }
    }
  }

  @Override
  public void clear() {
    pop.clear();
  }

  @Override
  public void evaluateAutomatic() {
    for (final T ind : pop) {
      if (!ind.isEvaluated()) {
        evaluator.evaluate(ind);
      }
    }
  }

  @Override
  public String toString() {
    final StringBuilder sb = new StringBuilder();

    for (final T ind : pop) {
      sb.append(ind);
      sb.append('\n');
    }

    if (sb.length() > 0) {
      sb.setLength(sb.length() - 1);
    }

    return sb.toString();
  }

  @Override
  public int size() {
    return pop.size();
  }

  @Override
  public T getUnfittestIndividual() {
    if (pop.size() == 0) {
      return null;
    }

    pop.sort(false);
    return pop.get(0);
  }

  @Override
  public T getFittestIndividual() {
    if (pop.size() == 0) {
      return null;
    }

    pop.sort(true);
    return pop.get(0);
  }

  @Override
  public T getEliteIndividual() {
    return getFittestIndividual();
  }

  @Override
  public T getRandomIndividualForEvaluation() {
    return pop.get(rnd.nextInt(pop.size()));
  }

  @Override
  public T getRandomIndividualForSelection() {
    return getRandomIndividualForEvaluation();
  }

  @Override
  public IndividualList<T> getUnevaluatedIndividuals() {
    final IndividualList<T> list = new IndividualList<T>();
    for (final T ind : pop) {
      if (!ind.isEvaluated()) {
        list.add(ind);
      }
    }

    return list;
  }

  @Override
  public boolean isEmpty() {
    return pop.isEmpty();
  }

  @Override
  public boolean isAllowDuplicates() {
    return allowDuplicates;
  }

  @Override
  public int getEvaluatedIndividualCount() {
    int i = 0;
    for (final T ind : pop) {
      if (ind.isEvaluated()) {
        i++;
      }
    }

    return i;
  }

  @Override
  public boolean containsAny(final IndividualList<T> list) {
    for (final T ind : list) {
      if (pop.contains(ind)) {
        return true;
      }
    }

    return false;
  }

  @Override
  public Iterator<T> iterator() {
    return pop.iterator();
  }

  @Override
  public int getInitIndividualCount() {
    return initIndividualCount;
  }

  @Override
  public void setInitIndividualCount(final int individualCount) {
    this.initIndividualCount = individualCount;
  }
}
